//       file: gargamel.asm (also creates azrael segment)
//    project: Turrican plain -TDE v1.0
// asm format: Kick Assembler
// ------------------------------------------------------
// replacement loader for original Turrican disk version "schlumpf.prg" 
// goal was to create a plain version from original Turrican disks without drive-code
// and without hidden encrypted files and without fast-loader code
// to be able to run it in an emulator without true drive emulation
// because it plays great with a gamepad on android, having space and f7 on gamepad
// to quickly access the bomb and wall shots 
// but switching disk images sucks in retroarch

// February 2019

* = $0326 // CHROUT vector address

.word prgstart                                  // autostart via kernal CHROUT vector hook, aka "READY" trap
.word $f6ed, $f13e, $f32f, $fe66, $f4a5, $f5ed  // other kernal vectors
memcfg: .byte $00 // for storing/restoring memory configuration

// information about files from original disks:
// final address indicates that the game relocates the loaded data to elsewhere in RAM
// 
/*                | original filename |           | temporary or  |
    new filename  | or loading call   | load addr | final address | notes / after load
    --------------+-------------------+-----------+---------------+-------------------
     0              IH                  6860-695B   C800,6860       high score table
     1              hidden              5000-8710   relocs itself   jsr $5F00
       1 relocates itself to $D000 and also to screen 3 in bank 3 and sets
       color RAM at $D800 as well (it's the fullscreen title picture)
     2              hidden              0B30-67FF   
       
     3              hidden              6800-C77F
       2 and 3 are the intro speech sample data, it can be heard
       but very distorted if imported as 8bit mono pcm between 2 or 4kHz in Audacity
       
     4              £A                  6960-6E97                   more drivecode
       4 gets loaded in original code at .C:c3f8, need to patch name to $C58F $C590
       then it jsr into it at $6C00 (seems to upload+execute more drivecode)
       this is what the "azrael" segment partly replaces and changes
       
     
     5              hidden              0400-685E
       5 loads during title picture "loading" message and the receiving
       loop loads it starting at 685E and growind downwards back to 0400
     
     6              hidden              6C00-C629                   jmp $6C00
       6 loads after "press space" was acknowledged
       contains mainscreen (with highscores and credits and music)
       (this also contains a personal message from "The Maniac" hidden in dump)
     
     7              hidden              A000-B5FF                   main sprites             
       7 loads after fire is pressed in mainscreen 
       the code re-uses the loader at 6960 and the call is in code in
       file 6 (so 6 needs/needed to be patched)
     
     8 originally disk side id file (obsolete now)
     
     9              hidden              E600-F479                   enemy sprites                                   
       9 loads during level 1-1
     A              hidden              7670-C7FF                   lvl 1-1 related 
     
     B to Y         hidden              various                     sprites, level-data 
       two files per level, one with sprites, one with level data
       locations vary: load address included in each file in first 2 bytes
     
     Z              hidden              6c00-bb36                   "the end"
*/


// Enables creation of breakpoints
//
.var brk_file = createFile("breakpoints.txt")

.macro break() 
{
  .eval brk_file.writeln("break " + toHexString(*))
}

.macro break_at(address)
{
  .eval brk_file.writeln("break " + toHexString(address))
}

.function address_of(symbol)
{
  .return "$" + toHexString(symbol,4)
}

.macro printpc(where)
{
  .print where + ": $" + toHexString(*,4)
}

.macro printpcexplizit(what, where)
{
  .print what + ": " + address_of(where)
}

.function getfname(number)
{
  .return fnames + number * 1
}

.function getfnameL(number)
{
  .return fnamesL + number * 1
}

.macro patchfname(target, number) // patch filename at target to filename by index
{
  .var at = getfname(number)
  lda at
  sta target
}

.macro patch(target, value)
{
  lda #value
  sta target
}

.macro patchJMP(target, jmpaddress)
{
  lda #JMP_ABS
  sta target
  lda #<jmpaddress
  sta target+1
  lda #>jmpaddress
  sta target+2
}

.macro patchJSR(target, jmpaddress)
{
  lda #JSR_ABS
  sta target
  lda #<jmpaddress
  sta target+1
  lda #>jmpaddress
  sta target+2
}

.macro patchaddress(target, address)
{
  lda #<address
  sta target
  lda #>address
  sta target+1
}

.macro patchNOP(target, count)
{
  ldx #<target
  ldy #>target
  lda #count
  jsr sr_nopper
}

.macro load0(number)
{
  .var at = getfname(number)
  ldx #<at
  ldy #>at
  jsr sr_load0
}

.macro loadL(number)
{
  .var at = getfnameL(number)
  ldx #<at
  ldy #>at
  jsr sr_loadB
}

// for use with fnames in early lookup table
// before loader is relocated to $6960
.macro patchCALLXY0(target, findex0)
{
  .var at = getfname(findex0) 
  ldx #<at
  stx target+1
  ldy #>at
  sty target+3
}

// for fnames from lookup table in azrael fnamesL
.macro patchCALLXYL(target, findexL)
{
  .var at = getfnameL(findexL) 
  ldx #<at
  stx target+1
  ldy #>at
  sty target+3
}


sr_load0: // replaces drive-code based loader with standard kernal loads
          // (changing this routine, requires changing schlumpf.asm as well
          //  because the mechanism relies on these being at same RAM locs)
printpc("sr_load0")
lda $01 
sr_load_sm_mcfg1:
sta memcfg // store ram/rom cfg
and #%11111000
ora #%00000110 // mem cfg to: RAM, IO, KERNAL
sta $01//*/

sm_fname_length:
lda #$01     //			SUBROUTINE_LOAD0 (load by filename as listed in directory)
jsr $FFBD    // KERNAL SET FILENAME, length A, Y high address, X low address
lda #$00
ldx #$08
ldy #$01
jsr $FFBA    // KERNAL SET LOGICAL FILE PARAMS, A file number, X device, Y secondary address
lda #$00
tax
tay
jsr $FFD5    // KERNAL LOAD/VERIFY RAM FROM DEVICE, A 0 = load, X,Y load address if secondary address == 0

sr_load_sm_mcfg2:
lda memcfg
sta $01 // restore pre load ram/rom cfg*/

rts    
sr_load_after:

.label fnameAZRAEL = *
.text "AZRAEL"

.label fnames = *
.text "0" 
.text "1" // title picture fullscreen
.text "2" // speech data
.text "3" // speech data
.text "4"
.text "5"
.text "6" // "mainscreen" loop (credits, scores)
.text "7" // main sprite and some goodies
//.text "8" // originall disk id info but I patched this to use as filename
          // lookup and var storage for relocated loader B at $6960
          // (obsolete since I removed disk side checks and placed file          
          //  lookup in azrael segment)


.label sr_loadB  = $6960 // location of Turrican loader B (from file "4")  
                         // will be replaced by "azrael" segment  
.label azrael_max = $6bff // azrael must not grow beyond this because
                          // this will get clobbered by later loadings
  
user_irq: // replacement interrupt handler to play nice with kernal
printpc("user_irq")
lda $D019
and #%10000001
beq no_rasterline_interrupt
jsr $c53a // inside original turrican interrupt handler (play music)
no_rasterline_interrupt:
//cli
jmp $EA31 // onward to kernal handler (will rti eventually)

jumped_from_c532: 
printpc("jumped_from_c532")
    // sei // was executed before the jmp in original code
    // X,Y already at lo,hi of filename "6"
    
    // restore power-on user interrupt handler at $0314 $0315
    lda #$31
    sta $0314
    lda #$ea
    sta $0315
    
    lda #$36 // select RAM, IO, KERNAL
    sta $01
       
    lda #$00
    sta $D019
    sta $D01A
    lda $D019 
    // raster compare interrupt cleared
    
    cli
    
    jsr sr_load0 // X,Y still point to RAM with "6" filename
    
    // the current routine "jumped_from_c532" must not grow beyond $03ff 
    // because file "5" will be loaded to $0400 before this executes
    // so more patch code needs to be placed in a different segment
    
    // load the extra segment, re-using RAM otherwise used by original
    // Turrican loaderB code which was already patched out
    lda #$06 // 6 letters in "azrael
    sta sm_fname_length+1
    ldx #<fnameAZRAEL
    ldy #>fnameAZRAEL
    jsr sr_load0 // load the extra segment below
    lda #$01     // back to 1 letter names
    sta sm_fname_length+1
    
    
    
.segment azrael [outPrg="azrael"]
printpcexplizit("sr_loadB (before more azrael routines)", sr_loadB)
* = sr_loadB
        lda $01 
        azrael_sr_load_sm_mcfg1:
        pha // sta azrael_sr_load_sm_mcfg2+1 // store ram/rom cfg
        and #%11111000
        ora #%00000110 // mem cfg to: RAM, IO, KERNAL
        sta $01//*/
        
        lda $9d
        pha     // remember zp
        
        lda #$00
        sta $9d // suppress kernal messages
        
        azrael_sm_fname_length:
        lda #$01     //			SUBROUTINE_LOAD0 (load by filename as listed in directory)
        jsr $FFBD    // KERNAL SET FILENAME, length A, Y high address, X low address
        lda #$00
        ldx #$08
        ldy #$01
        jsr $FFBA    // KERNAL SET LOGICAL FILE PARAMS, A file number, X device, Y secondary address
        lda #$00
        tax
        tay
        jsr $FFD5    // KERNAL LOAD/VERIFY RAM FROM DEVICE, A 0 = load, X,Y load address if secondary address == 0
        
        pla
        sta $9d // restore zp 
        
        azrael_sr_load_sm_mcfg2:
        pla
        sta $01 // restore pre load ram/rom cfg*/
        
        rts    
        azrael_sr_load_after:
azrael:
printpc("azrael")
    jmp azrael_patch_ahead
    
    printpc("sr_nopper (in azrael)")
    sr_nopper:
    stx $02
    sty $03
    tay
    dey
    lda #NOP
    lnop:
    sta ($02),y
    dey
    bpl lnop
    rts
    
    // patches for IRQ handler used during loading screen
    printpc("(in azrael) fnamesL")
    fnamesL: .text "79A" // BCDEFGHIJKLMNOPQRSTUVWXYZ" // levels and ending
    fnameCUR: .text "?"  // will be changed at runtime
    
    azrael_irq: // replacement interrupt handler to play nice with kernal
                // during first level loading
    printpc("azrael_irq")
    pha
    
    lda $01
    and #%00000111
    cmp #$04
    bne azrael_irq_ok // if IO available
    
    pla
    rti
    azrael_irq_ok:
            lda $d019
            and #%10000001
            beq azrael_after_raster_irq
    
    txa
    pha
    tya
    pha
    .var azrael_line_irq_first = 114 // during first level loading
    .var azrael_line_irq_other = 106 // during other level and the end loading
            azrael_irq_sm_line:
            lda #azrael_line_irq_first 
            !: cmp $d012
            bpl !-
            
    .var azrael_loader_bg_sub_first = $6cb9 // during first
    .var azrael_loader_bg_sub_other = $62be // during other
            //sei
            azrael_irq_sm_sub:
            // don't insert anything in this line! :P
            jsr azrael_loader_bg_sub_first // draws the bg lines behind msg
            dec $d019 // ack
            //cli
            
    pla
    tay
    pla
    tax
            lda $01
            and #%00000111
            cmp #$06
            beq azrael_after_raster_irq // if kernal available
    pla
    azrael_irq_just_rti:
    rti
    azrael_after_raster_irq:
    pla
    jmp $EA31 // onward to kernal handler (will rti eventually)
 

    azrael_irq_tail: // needed for interrupt handler as subroutine
                     // during loadings between levels
    lda #$00
    sta $D020
    azrael_irq_just_rts:
    rts


                
   
    azrael_patch_loader:
    printpc("azrael_patch_loader")    
      sei
      lda #$36
      sta $01 // make sure kernal is on in case cli immediately fires an interrupt
     
      lda #%01111111
      sta $DC0D
      sta $DD0D
      lda $DC0D
      lda $DD0D
      lda #%10000001
      sta $DC0D
      lda $DC0D // power-on interrupt IRQ state restored (need for kernel load)
      
      lda #$00  // disable raster compare interrupt
      sta $D019
      sta $D01A
      lda $D019 
            
      /*lda #<azrael_irq_just_rti
      sta $fffa
      lda #>azrael_irq_just_rti
      sta $fffb*/ // turrican already does sth. like this
      
      // set default kernal irq handler
      lda #$48
      sta $FFFE
      lda #$FF
      sta $FFFF
      
      // set user irq handler
      lda #<azrael_irq
      sta $0314
      lda #>azrael_irq
      sta $0315
      
      lda #$01
      sta $d019 // request raster irq..
      sta $d01A // request interrupt by rasterline only
      lda $d011
      and #$7f
      sta $d011 // clear raster cmp high
      azrael_loader_sm_line:
      lda #azrael_line_irq_first-3  // ..at this line (bit earlier than needed)
      sta $d012
      
    rts
    


    
    azrael_load7:
    // replaces original setup to skip the "turn disk" screen
    printpc("azrael_load7")
    // still "sei" from azrael_patch_loader_after_main_screen
    
    lda #$35 // RAM, IO, RAM
    sta $01
    ldy #$01
    jsr $6c60 // prep loading screen step 1, rts patched at .C:6c77
              // prepares screen and color ram
    jsr $78cc // prep loading screen step 2: writes loading msg   
              // step 3
    jsr $7901 // modifies "turn disk gray bg" irq routine to blue bg "loading"
              // also turns screen on again
     
    ldx #<getfnameL(0)
    ldy #>getfnameL(0)
    jsr sr_loadB // might cli during kernal calls
    
    sei
    lda #$34 // RAM, RAM, RAM
    sta $01
    
    // relocates content from "7" to under IO area
    jsr $6c4a // returns at $6c5d, because RTS was patched in before 
    
    lda #$36 // RAM, IO, KERNAL
    sta $01
    
    jmp $6c9f // proceed with loading "9"
    //rts
    
    
    
    
    azreal_patch_after_load9: // patched in at 6ca6
    printpc("azreal_patch_after_load9")
    
    loadL(2) // "A" level 1-1 data
    jsr azrael_after_loaderB
    jmp azrael_after_loaderB_extra // will rts from current routine
    
    azrael_after_loaderB:
    sei
    
    // restore turrican state for code line C:6ca9 (before jmp to $08e3)
    // (examined locations and values by breaking at original code before jumped)
    lda #$35 // RAM, IO, RAM
    sta $01
    
    // restore 0314
    lda #$31
    sta $0314
    lda #$ea
    sta $0315
    
    lda #$00  // disable raster compare interrupt
    sta $D019
    sta $D01A
    lda $D019 
    //inc $d019 // enable    
    
    lda #$7f
    sta $dc0d // clear
    lda $dc0d
    sta $dd0d
    lda $dd0d
    
    lda #$81  // set
    sta $dc0d
    rts
    
    azrael_after_loaderB_extra:    
    jsr $6CE8 // SUBROUTINE_SET_RASTER_CMP_LOW_12DEC (waits for line -1 first)
    rts
    
    azrael_load_fixed_fname:
    ldx #<fnameCUR
    ldy #>fnameCUR
    jsr sr_loadB
    rts
    
    
    azrael_load_next_level:
    lda $03fc // turrican level index
    asl // * 2
    clc
    adc #('B'-2) // -2 because level index is at least 1 when we get here
    
    sta fnameCUR
    cmp #'Z' // check for "the end"?
    beq azrael_just_load_the_end
 
    jsr azrael_load_fixed_fname // load sprites for next level
    
    inc fnameCUR
    azrael_just_load_the_end:
    jsr azrael_load_fixed_fname // load level data for next level or end screen
    
    rts
    
    
    
    
    azrael_loader_between_levels: // injected at 099a
    sei
    // make changes to azrael_irq
        // use other line for interrupt
        patch(azrael_loader_sm_line+1, azrael_line_irq_other-3)
        patch(azrael_irq_sm_line+1, azrael_line_irq_other)

        // use other subroutine for drawing the banner behind loading text
        patchaddress(azrael_irq_sm_sub+1, azrael_loader_bg_sub_other)
    
        jsr $62EC // interrupt setup and screen settings for loading screen
    
    // save kernal zp area (kernal load will clobber it)
    ldx #$FD
    !:
    lda $0002,x
    sta $CC00,x // use screen 3, bank 3 as storage (unused by turrican atm)
    //lda $0202,x
    //sta $CE00,x
    dex
    bpl !-

    jsr azrael_patch_loader // prepare loading
    
    jsr azrael_load_next_level // do the loading
    
    sei
    
    // revert changes to azrael_irq
        // restore line for interrupt
        patch(azrael_loader_sm_line+1, azrael_line_irq_first-3)
        patch(azrael_irq_sm_line+1, azrael_line_irq_first)

        // restore subroutine for drawing the banner behind loading text
        patchaddress(azrael_irq_sm_sub+1, azrael_loader_bg_sub_first)
    
    jsr azrael_after_loaderB 
        
    
    // restore kernal zp area (kernal load clobbered it)
    ldx #$FD
    !:
    lda $CC00,x // used screen 3, bank 3 as storage (unused by turrican atm)
    sta $0002,x
    //lda $CE00,x
    //sta $0202,x
    dex
    bpl !-

    rts
    
    
    
    
    azrael_reload_and_repatch_main:
    lda #('6') // file "6" contains the mainscreen code
    sta fnameCUR
    jsr azrael_after_loaderB // disable all interrupts
    jsr azrael_load_fixed_fname
    sei
    jsr azrael_patch_ahead // patch all the calls again inside the reloaded code
    rts
    
    
    
    
    azrael_patch_ahead:
    printpc("azrael_patch_ahead")
    patch($6ce1, RTS) // modify original interrupt handler into subroutine
        
    // before continuing patch files 7 and up in at appropriate locs
    // patch calls at $6c40 to prepare for upcoming loadings
    patchJSR($6c40, azrael_patch_loader)
    patchJMP($6c43, azrael_load7)
    
    // remove "turn the disk" message and spacebar check loop
    patchNOP($6C77, 3) // remove call which would prepare "turn the disk"
    
    patchNOP($6C7D, 7) // remove wait for spacebar loop
        
    // patch call to load file "9"
    patchCALLXYL($6C9F, 1) // index 1 for lvl 1-1 sprites
    
    // replace jsr $6ce8 at .C:6ca6 with after_load9 call
    patchaddress($6ca7, azreal_patch_after_load9)
    
    // patch ahead, create subroutine out of relocation code after "7"
    patch($6c5d, RTS)
    // patch ahead, create subroutine out of screen preps after "7"
    patch($6c77, RTS)
    
    // patch raster compare line for first level load interrupt handler
    patch($790c, azrael_line_irq_first-3)
    
    patch($791f, RTS) // shortcut loading screen setup step 3
    
    // remove original calls for loadscreen preps
    // and remove wait for specific rasterline
    // also remove disk side check
    patchNOP($6c8b, 20)
    
    
    // --- patch calls and addresses for loader between levels ---
    // inject custom loader routine for inbetween level loadings
    patchaddress($099b, azrael_loader_between_levels)
    
    // return early from interrupt setup routine and remove cli there
    patch($62ff, RTS)
    
    // turn original interrupt handler into subroutine
    patchaddress($62df, azrael_irq_tail) // replacement tail for that handler
    
    // patch raster compare line for interrupt handler
    patch($62fb, azrael_line_irq_other-3)
    
    patchNOP($099d, 7) // remove original call to turrican loaderB
    
    
    // --- patch calls to loader after gameloop ends ---
    patchJMP($0935, $0957) // skip "turn disk" screen after game ends
    patchaddress($0968, azrael_reload_and_repatch_main)
    
    
    // --- patch scoring related calls ---
    // eliminate loader de-initialization call, because it is only
    // meaningful if the drive was running original turrican loader drive-code
    lda #BIT_ABS
    sta $6fd6    
    
     // skip "insert side 1 of your gamedisk" screen before saving highscore
    patchJMP($728f, $729a)
    // and skip the check for run/stop or fire button after music fade
    patchJMP($729d, $72a9)
    
    // elimintate drive-code send&execute routine before saving scores   
    patchNOP($74d0, 9)
    // A still holds NOP
    // fix filename to scratch before saving scores
    lda #('0') 
    sta $74ee
    patchNOP($74f2, 5)
    patch($7509, 1) // 1 char filename length
    patch($750b, $ee) // lo of filename address
    patch($750d, $74) // hi of filename address
    
    rts
    
// this segment itself must not grow out of 6960-6bff or it will be clobbered    
// by subsequent loadings     
.if (* > azrael_max) .print "-------- ERROR: azrael has grown too fat! --------"
printpc("after azrael") 

.segment Default    

    
    jsr azrael  // call the just loaded segment routine for more patches
    
    // the mainscreen which runs after the next rts
    // already clobbers low RAM locations, among them sr_nopper, so
    // all code patches must be ran ahead of time in inside jsr azrael
    
    // stack return address actually points at $6C00 at this point so
    // can just rts into loaded game mainscreen
    
    rts
    // should never reach this line
    jmp $6C00 // start address of code in "6"
    

.label prgstart = *
printpc("main")
main: 
    sei
// ---- section copied verbatim from original schlumpf.prg -------------------------------------------------------
    cld
    lda #$CA
    sta $0326
    lda #$F1
    sta $0327 // original CHROUT vector restored
     
    lda #$00
    sta $D120 // BORDER COLOR BLACK	(VIC registers at $D000 repeat every 64 bytes until $D3FF)
    sta $D221 // BACKGROUND COLOR BLACK
    sta $D1D1 // SCREEN CONTROL REGISTER (same as $D011, because of repeated registers at 00,40,80,C0 (D1-C0==11)
              // v-scroll 0, screen height = 24 rows, screen off, extended bg mode off, raster interrupt line 0
    sta $D015 // disable sprites
    sta $9D   // suppress error/kernal messages
// ---- end of section copied verbatim from original schlumpf.prg ------------------------------------------------    
    
    load0(0) // "IH"
    sei
    ldx #$FF
    txs 
    ldx #$00          // FF bytes copy loop COPY_HIGHSCORE_LOOP
copyscores: printpc("copyscores")
    lda $6860,x       // from loaded highscores  $6860..$6959 (F9 bytes total)
    sta $C800,x       // to $C800
    inx
    bne copyscores    // after loop: copied highscore table (and the seven bytes beyond it) to $C800    
    
    jmp show_readthis
fnameR:                 
.text "READTHIS"
show_readthis:
    lda #$08
    sta sm_fname_length+1 // filename length to 8
    
    ldx #<fnameR
    ldy #>fnameR
    jsr sr_load0 // that file loads to screen 3 in bank 1
    
    sei
    lda #$01
    sta sm_fname_length+1 // filename length back to 1
    
    lda #$37
    sta $01 // basic, IO, kernal to active
    
    lda #DARK_GRAY
    sta $D020
    sta $D021
    lda #LIGHT_GRAY
    ldx #$00
    !:
    sta $d800,x
    sta $d900,x
    sta $da00,x
    sta $db00,x
    inx
    bne !- // set color ram
    
    // save current vic memory control reg on stack
    lda $d018
    pha
    
    // activate screen and charset 3 for displaying loaded READTHIS content
    lda #%00110111 // screen 3, charset 3
    sta $d018  
    lda #%00011011 // textmode, screen on, 25 rows, scroll-y 3
    sta $d011
    
    !:
    lda $dc01
    and #$10
    bne !- // wait for key-down: space key
    !:
    lda $dc01
    and #$10
    beq !- // wait for key-up: space key

    lda #$00
    sta $d011 // screen off

    // restore vic memory control reg
    pla
    sta $d011
     
    lda #$00
    sta $D120 // BORDER COLOR BLACK	(VIC registers at $D000 repeat every 64 bytes until $D3FF)
    sta $D221 // BACKGROUND COLOR BLACK
    sta $D1D1 // SCREEN CONTROL REGISTER (same as $D011, because of repeated registers at 00,40,80,C0 (D1-C0==11)
              // v-scroll 0, screen height = 24 rows, screen off, extended bg mode off, raster interrupt line 0
    
    load0(1)
    jsr $5F00 // initializes and relocates what was just loaded (title picture, colors, sprites)
    
    load0(2)
    load0(3)
    
    // before proceeding to jump into loaded code $C330, need to patch things
    
    // to skip over short section after "HELLO AND WELCOME..."
    // (would otherwise send some direct commands to drive)
    patchJMP($C3DD, $C3F2)         
    
    patchfname($C58F, 4) // change "£A" to "4"
    
    // .C:c40b  20 00 6C    JSR $6C00 // sends more drive-code over serial bus
    // nop out that drive-code sending routine
    lda #NOP
    sta $c40b
    sta $c40c
    sta $c40d
    
    
    // replace drive-code-loader-B based load at 
    // .C:c4e9  A2 2B       LDX #$2B
    // .C:c4eb  A0 30       LDY #$30
    // .C:c4ed  20 60 69    JSR $6960
    patch($c4ea, <getfname(5))
    patch($c4ec, >getfname(5))
    patch($c4ee, <sr_load0)
    patch($c4ef, >sr_load0)
    
    
    // replace interrupt handler during title picture
    // .C:c44a  78          SEI
    // .C:c44b  A9 35       LDA #$35
    // .C:c44d  8D FE FF    STA $FFFE
    // .C:c450  A9 C5       LDA #$C5
    // .C:c452  8D FF FF    STA $FFFF
    patch($c44c, <user_irq)
    patch($c451, >user_irq)
    patchaddress($c44e, $0314)
    patchaddress($c453, $0315)
    patch($c54e, RTS) // make original turrican handler return normally
    
    // patch memory selection to RAM, IO, KERNAL for kernal load
    // during title picture
    patch($c479, LDA_IMM)
    patch($c47A, $36) 
    patch($c47B, STA_ZP)
    patch($c47C, $01)
    patch($c47D, $EA)
    patch($c47E, $EA)
    
    // replace drive-code-loader-B based load at
    // .C:c52d  78          SEI
    // .C:c52e  A2 34       LDX #$34
    // .C:c530  A0 25       LDY #$25
    // .C:c532  4C 60 69    JMP $6960	$6960		loads more stuff call SUBROUTINE_LOADER_B0
    //patch($c52d, NOP)
    patch($c52f, <getfname(6))
    patch($c531, >getfname(6))
    patchJMP($c532,jumped_from_c532) 
    
    title_pic_and_music:
    jmp $C330	//	proceed with "HELLO AND WELCOME TO TURRICAN..." speech
    // music and title picture now playing while next file is loaded
    // "press space" appears when it is loaded
    
    // problem was, replacement loader code grows into 
    // 0400 where it needs to
    // load other stuff, so I moved main down, kept irq earlier
    // so main can be clobbered by loader after patching the original code 
    // has finished at runtime
    
    // rts // this will be clobbered anyway.. :P
printpc("end of code")
 